﻿using System;
using System.Configuration;
using System.Data.EntityClient;
using System.Data.Objects;
using System.Linq;
using System.Transactions;

namespace EfExamples
{
    class Program
    {
        static void Main(string[] args)
        {
            // Listing 14-3. Używanie wygenerowanych typów encji
            using (var dbContext = new AdventureWorksLT2008Entities())
            {
                DateTime orderDate = new DateTime(2004, 6, 1);
                var orders = from order in dbContext.SalesOrderHeaders
                             where order.OrderDate == orderDate
                             select order;
                foreach (SalesOrderHeader order in orders)
                {
                    Console.WriteLine(order.TotalDue);
                }
            }

            // Listing 14-4. Używanie właściwości nawigacji
            using (var dbContext = new AdventureWorksLT2008Entities())
            {
                var customerOrderCounts = from cust in dbContext.Customers
                                          select new
                                          {
                                              cust.CustomerID,
                                              OrderCount = cust.SalesOrderHeaders.Count
                                          };
                foreach (var customerInfo in customerOrderCounts)
                {
                    Console.WriteLine("Klient {0} ma {1} zamówień",
                    customerInfo.CustomerID, customerInfo.OrderCount);
                }
            }

            using (var dbContext = new AdventureWorksLT2008Entities())
            {
                // Listing 14-5. Korzystanie z różnych związków przy użyciu właściwości nawigacji
                var info = from cust in dbContext.Customers
                           select new
                           {
                               cust.CustomerID,
                               Orders = from order in cust.SalesOrderHeaders
                                        where order.Status == 5
                                        select new
                                        {
                                            order.TotalDue,
                                            ItemCount = order.SalesOrderDetails.Count
                                        }
                           };

                foreach (var item in info)
                {
                    if (item.Orders.Count() > 0)
                    {
                        Console.WriteLine(item.CustomerID);
                        foreach (var order in item.Orders)
                        {
                            Console.WriteLine(" " + order);
                        }
                    }
                }

                {
                    // Listing 14-6. Używanie asocjacji bezpośrednio po wstępnym zapytaniu
                    Customer myCustomer = dbContext.Customers.Single(
                        cust => cust.CustomerID == 29531);
                    Console.WriteLine(myCustomer.SalesOrderHeaders.Count);
                }

                {
                    // Listing 14-7. Jawne ładowanie encji dla asocjacji
                    Customer myCustomer = dbContext.Customers.Single(
                    cust => cust.CustomerID == 29531);
                    myCustomer.SalesOrderHeaders.Load();
                    Console.WriteLine(myCustomer.SalesOrderHeaders.Count);
                }

                {
                    // Listing 14-8. Określanie związków, które mają zostać wstępnie załadowane z bazy
                    var customersWithOrderDetails = dbContext.Customers.
                        Include("SalesOrderHeaders.SalesOrderDetails");
                    Customer myCustomer = customersWithOrderDetails.Single(
                        cust => cust.CustomerID == 29531);
                    Console.WriteLine(myCustomer.SalesOrderHeaders.Count);
                }


                {
                    DateTime orderDate = new DateTime(2004, 6, 1);

                    // Listing 14-9. Proste wyrażenie zapytania LINQ to Entities
                    var orders = from order in dbContext.SalesOrderHeaders
                                 where order.OrderDate == orderDate
                                 select order;

                    // Zapytanie nie jest naprawdę wykonywane w tym momencie.
                    // Aby jakieś rzeczywiste działania zostały przeprowadzone,
                    // należy przystąpić do pobierania z niego elementów.
                    
                    // Nawet ten kod nie wykonuje zapytania 'orders',
                    // mimo że odwołuje się do niego - buduje nowe zapytanie,
                    // które stanowi rozszerzoną wersję oryginalnego.

                    // Listing 14-10. Zapytanie łańcuchowe
                    var orderedOrders = from order in orders
                                        orderby order.OrderDate
                                        select order;
                }

                {
                    DateTime orderDate = new DateTime(2004, 6, 1);

                    // Listing 14-11. To samo zapytanie w pełnej postaci
                    var orderedOrders = from order in dbContext.SalesOrderHeaders
                                        where order.OrderDate == orderDate
                                        orderby order.OrderDate
                                        select order;
                }
            }

            // Listing 14-12. Użycie zapytania ESQL
            using (var dbContext = new AdventureWorksLT2008Entities())
            {
                DateTime orderDate = new DateTime(2004, 6, 1);
                var query = dbContext.CreateQuery<SalesOrderHeader>("SELECT VALUE o " +
                        "FROM AdventureWorksLT2008Entities.SalesOrderHeaders AS o " +
                        "WHERE o.OrderDate = @orderDate",
                    new ObjectParameter("orderDate", orderDate));

                foreach (var order in query)
                {
                    Console.WriteLine(order.TotalDue);
                }
            }

            // Listing 14-14. Przekazywanie jawnego łańcucha połączenia

            // Pobieranie łańcucha połączenia dla podstawowego dostawcy bazy danych
            ConnectionStringSettings dbConnectionInfo =
                ConfigurationManager.ConnectionStrings["AdventureWorksSql"];

            var csb = new EntityConnectionStringBuilder();
            csb.Provider = dbConnectionInfo.ProviderName;
            csb.ProviderConnectionString = dbConnectionInfo.ConnectionString;
            csb.Metadata = "res://*/AdventureWorksModel.csdl|" +
                "res://*/AdventureWorksModel.ssdl|res://*/AdventureWorksModel.msl";

            using (var dbContext = new AdventureWorksLT2008Entities(csb.ConnectionString))
            {
                foreach (SalesOrderHeader order in dbContext.SalesOrderHeaders)
                {
                    Console.WriteLine(order.TotalDue);
                }
            }


            // Listing 14-16. Modyfikowanie istniejącej encji
            using (var dbContext = new AdventureWorksLT2008Entities())
            {
                var orderQuery = from customer in dbContext.Customers
                                 where customer.CustomerID == 29531
                                 from order in customer.SalesOrderHeaders
                                 orderby order.OrderDate descending
                                 select order;
                SalesOrderHeader latestOrder = orderQuery.First();
                latestOrder.Comment = "Zadzwonić do klienta, gdy produkty będą gotowe do wysyłki";
                dbContext.SaveChanges();
            }

            try
            {
                using (var dbContext = new AdventureWorksLT2008Entities())
                {
                    // Listing 14-17. Błąd polegający na niespełnieniu wymagań narzucanych na nową encję przez więzy
                    SalesOrderDetail detail = new SalesOrderDetail();
                    dbContext.AddToSalesOrderDetails(detail);
                    // Ta instrukcja spowoduje zgłoszenie wyjątku!
                    dbContext.SaveChanges();
                }
            }
            catch (Exception x)
            {
                Console.WriteLine(x);
            }

            using (var dbContext = new AdventureWorksLT2008Entities())
            {
                var orderQuery = from customer in dbContext.Customers
                                 where customer.CustomerID == 29531
                                 from order in customer.SalesOrderHeaders
                                 orderby order.OrderDate descending
                                 select order;
                SalesOrderHeader latestOrder = orderQuery.First();


                // Listing 14-18. Dodawanie nowej encji
                // ...gdzie latestOrder to SalesOrderHeader pobrany za pomocą kodu podobnego do tego,
                // który został przedstawiony na listingu 14.16.
                SalesOrderDetail detail = new SalesOrderDetail();
                detail.SalesOrderHeader = latestOrder;
                detail.ModifiedDate = DateTime.Now;
                detail.OrderQty = 1;
                detail.ProductID = 680; // HL Road Frame - Black, 58
                dbContext.AddToSalesOrderDetails(detail);
                dbContext.SaveChanges();

                // Listing 14-19. Usuwanie encji
                dbContext.DeleteObject(detail);
                dbContext.SaveChanges();
            }

            // Listing 14-20. Zastosowanie klasy TransactionScope
            using (var dbContext = new AdventureWorksLT2008Entities())
            {
                using (var txScope = new TransactionScope())
                {
                    var customersWithOrders = from cust in dbContext.Customers
                                              where cust.SalesOrderHeaders.Count > 0
                                              select cust;
                    foreach (var customer in customersWithOrders)
                    {
                        Console.WriteLine("Klient {0} ma {1} zamówień",
                        customer.CustomerID, customer.SalesOrderHeaders.Count);
                    }
                    txScope.Complete();
                }
            }
        }
    }
}
